"""Constants for the Passive BLE monitor integration."""
from __future__ import annotations

from dataclasses import dataclass

from homeassistant.components.binary_sensor import (
    BinarySensorDeviceClass, BinarySensorEntityDescription)
from homeassistant.components.sensor import (SensorDeviceClass,
                                             SensorEntityDescription,
                                             SensorStateClass)
from homeassistant.const import (CONCENTRATION_MICROGRAMS_PER_CUBIC_METER,
                                 CONCENTRATION_MILLIGRAMS_PER_CUBIC_METER,
                                 CONCENTRATION_PARTS_PER_MILLION, CONDUCTIVITY,
                                 LIGHT_LUX, PERCENTAGE,
                                 SIGNAL_STRENGTH_DECIBELS_MILLIWATT, Platform,
                                 UnitOfElectricPotential, UnitOfEnergy,
                                 UnitOfMass, UnitOfPower, UnitOfPressure,
                                 UnitOfTemperature, UnitOfVolume)
from homeassistant.helpers.entity import EntityCategory

DOMAIN = "ble_monitor"
PLATFORMS = [
    Platform.DEVICE_TRACKER,
    Platform.BINARY_SENSOR,
    Platform.SENSOR,
]

# Configuration options
CONF_BT_AUTO_RESTART = "bt_auto_restart"
CONF_PERIOD = "period"
CONF_LOG_SPIKES = "log_spikes"
CONF_USE_MEDIAN = "use_median"
CONF_ACTIVE_SCAN = "active_scan"
CONF_HCI_INTERFACE = "hci_interface"
CONF_BT_INTERFACE = "bt_interface"
CONF_BATT_ENTITIES = "batt_entities"
CONF_REPORT_UNKNOWN = "report_unknown"
CONF_RESTORE_STATE = "restore_state"
CONF_DEVICE_ENCRYPTION_KEY = "encryption_key"
CONF_DEVICE_USE_MEDIAN = "use_median"
CONF_DEVICE_REPORT_UNKNOWN = "report_unknown"
CONF_DEVICE_RESTORE_STATE = "restore_state"
CONF_DEVICE_RESET_TIMER = "reset_timer"
CONF_DEVICE_TRACK = "track_device"
CONF_DEVICE_TRACKER_SCAN_INTERVAL = "tracker_scan_interval"
CONF_DEVICE_TRACKER_CONSIDER_HOME = "consider_home"
CONF_DEVICE_DELETE_DEVICE = "delete device"
CONF_PACKET = "packet"
CONF_GATEWAY_ID = "gateway_id"
CONF_UUID = "uuid"
CONFIG_IS_FLOW = "is_flow"

SERVICE_CLEANUP_ENTRIES = "cleanup_entries"
SERVICE_PARSE_DATA = "parse_data"

# Default values for configuration options
DEFAULT_BT_AUTO_RESTART = False
DEFAULT_PERIOD = 60
DEFAULT_LOG_SPIKES = False
DEFAULT_USE_MEDIAN = False
DEFAULT_ACTIVE_SCAN = False
DEFAULT_BATT_ENTITIES = True
DEFAULT_REPORT_UNKNOWN = "Off"
DEFAULT_DISCOVERY = True
DEFAULT_RESTORE_STATE = False
DEFAULT_DEVICE_MAC = ""
DEFAULT_DEVICE_UUID = ""
DEFAULT_DEVICE_ENCRYPTION_KEY = ""
DEFAULT_DEVICE_USE_MEDIAN = "default"
DEFAULT_DEVICE_REPORT_UNKNOWN = False
DEFAULT_DEVICE_RESTORE_STATE = "default"
DEFAULT_DEVICE_RESET_TIMER = 35
DEFAULT_DEVICE_TRACKER_SCAN_INTERVAL = 20
DEFAULT_DEVICE_TRACKER_CONSIDER_HOME = 180
DEFAULT_DEVICE_TRACK = False
DEFAULT_DEVICE_DELETE_DEVICE = False

# regex constants for configuration schema
MAC_REGEX = "(?i)^(?:[0-9A-F]{2}[:]){5}(?:[0-9A-F]{2})$"
# MiBeacon V2/V3 uses 24 character long key
AES128KEY24_REGEX = "(?i)^[A-F0-9]{24}$"
# MiBeacon V4/V5 uses 32 character long key
AES128KEY32_REGEX = "(?i)^[A-F0-9]{32}$"

# Fixed constants

# Sensor measurement limits to exclude erroneous spikes from the results (temperature in °C)
CONF_TMIN = -40.0
CONF_TMAX = 85.0
CONF_TMIN_KETTLES = -20.0
CONF_TMAX_KETTLES = 120.0
CONF_TMIN_PROBES = 0.0
CONF_TMAX_PROBES = 300.0
CONF_HMIN = 0.0
CONF_HMAX = 99.9


# Sensors with deviating temperature range
KETTLES = ('YM-K1501', 'YM-K1501EU', 'V-SK152')
PROBES = ('iBBQ-2', 'iBBQ-4', 'iBBQ-6', 'H5182', 'H5183', 'H5184', 'H5185', 'H5198')


# Sensor entity description
@dataclass
class BLEMonitorRequiredKeysMixin:
    """Mixin for required keys."""

    sensor_class: str
    update_behavior: str
    unique_id: str


@dataclass
class BLEMonitorSensorEntityDescription(
    SensorEntityDescription, BLEMonitorRequiredKeysMixin
):
    """Describes BLE Monitor sensor entity."""


@dataclass
class BLEMonitorBinarySensorEntityDescription(
    BinarySensorEntityDescription, BLEMonitorRequiredKeysMixin
):
    """Describes BLE Monitor binary sensor entity."""


BINARY_SENSOR_TYPES: tuple[BLEMonitorBinarySensorEntityDescription, ...] = (
    BLEMonitorBinarySensorEntityDescription(
        key="battery charging",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="battery charging",
        unique_id="bc_",
        device_class=BinarySensorDeviceClass.BATTERY_CHARGING,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="battery low",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="battery low",
        unique_id="bl_",
        device_class=BinarySensorDeviceClass.BATTERY,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="carbon monoxide",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="carbon monoxide",
        unique_id="bco_",
        device_class=BinarySensorDeviceClass.CO,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="cold",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="cold",
        unique_id="bcold_",
        device_class=BinarySensorDeviceClass.COLD,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="connectivity",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="connectivity",
        unique_id="bconn_",
        device_class=BinarySensorDeviceClass.CONNECTIVITY,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="door",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble door",
        unique_id="do_",
        device_class=BinarySensorDeviceClass.DOOR,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="garage door",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="garage door",
        unique_id="bgdr_",
        device_class=BinarySensorDeviceClass.GARAGE_DOOR,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="gas_detected",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="gas",
        unique_id="bgas_",
        device_class=BinarySensorDeviceClass.GAS,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="heat",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="heat",
        unique_id="bheat_",
        device_class=BinarySensorDeviceClass.HEAT,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="light",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble light",
        unique_id="lt_",
        device_class=BinarySensorDeviceClass.LIGHT,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="lock",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble lock",
        unique_id="lock_",
        device_class=BinarySensorDeviceClass.LOCK,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="moisture detected",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble moisture",
        unique_id="mo_",
        device_class=BinarySensorDeviceClass.MOISTURE,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="motion",
        sensor_class="MotionBinarySensor",
        update_behavior="Instantly",
        name="ble motion",
        unique_id="mn_",
        device_class=BinarySensorDeviceClass.MOTION,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="moving",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="moving",
        unique_id="bmove_",
        device_class=BinarySensorDeviceClass.MOVING,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="occupancy",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="occupancy",
        unique_id="bocc_",
        device_class=BinarySensorDeviceClass.OCCUPANCY,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="opening",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble opening",
        unique_id="op_",
        device_class=BinarySensorDeviceClass.OPENING,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="plug",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble plug",
        unique_id="plug_",
        device_class=BinarySensorDeviceClass.PLUG,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="presence",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="presence",
        unique_id="presence_",
        device_class=BinarySensorDeviceClass.PRESENCE,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="problem",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="problem",
        unique_id="problem_",
        device_class=BinarySensorDeviceClass.PROBLEM,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="running",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="running",
        unique_id="running_",
        device_class=BinarySensorDeviceClass.RUNNING,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="safety",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="safety",
        unique_id="safety_",
        device_class=BinarySensorDeviceClass.SAFETY,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="smoke",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble smoke",
        unique_id="sd_",
        device_class=BinarySensorDeviceClass.SMOKE,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="sound",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="sound",
        unique_id="sound_",
        device_class=BinarySensorDeviceClass.SOUND,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="tamper",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="tamper",
        unique_id="tamper_",
        device_class=BinarySensorDeviceClass.TAMPER,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="vibration",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble vibration",
        unique_id="vi_",
        icon="mdi:vibrate",
        device_class=BinarySensorDeviceClass.VIBRATION,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="window",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="window",
        unique_id="window_",
        device_class=BinarySensorDeviceClass.WINDOW,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="binary",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble binary",
        unique_id="bi_",
        device_class=None,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="remote single press",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble remote binary single press",
        unique_id="rb_single_press_",
        device_class=None,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="remote long press",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble remote binary long press",
        unique_id="rb_long_press_",
        device_class=None,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="weight removed",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble weight removed",
        icon="mdi:weight",
        unique_id="wr_",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="fingerprint",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble fingerprint",
        icon="mdi:fingerprint",
        unique_id="fp_",
        device_class=None,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="antilock",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble antilock",
        unique_id="antilock_",
        device_class=BinarySensorDeviceClass.LOCK,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="childlock",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble childlock",
        unique_id="childlock_",
        device_class=BinarySensorDeviceClass.LOCK,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="armed away",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble armed away",
        unique_id="armedaway_",
        device_class=BinarySensorDeviceClass.LOCK,
        force_update=True,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="toothbrush",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble toothbrush",
        unique_id="tb_",
        icon="mdi:toothbrush-electric",
        device_class=BinarySensorDeviceClass.POWER,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="tilt",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble tilt",
        unique_id="ti_",
        icon="mdi:rotate-orbit",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="dropping",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble dropping",
        unique_id="dr_",
        icon="mdi:chevron-triple-down",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="impact",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble impact",
        unique_id="imp_",
        icon="mdi:arrow-collapse-right",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="magnetic field detected",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="magnetic field",
        unique_id="magnetic_field_",
        icon="mdi:magnet",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="bed occupancy",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="bed occupancy",
        unique_id="bed_occ_",
        icon="mdi:bed",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="snoring",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="snoring",
        unique_id="snoring_",
        icon="mdi:sleep",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="sleeping",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="sleeping",
        unique_id="sleeping_",
        icon="mdi:sleep",
        device_class=None,
        force_update=False,
    ),
    BLEMonitorBinarySensorEntityDescription(
        key="switch",
        sensor_class="BaseBinarySensor",
        update_behavior="Instantly",
        name="ble switch",
        unique_id="sw_",
        device_class=BinarySensorDeviceClass.POWER,
        force_update=True,
    ),
)


SENSOR_TYPES: tuple[BLEMonitorSensorEntityDescription, ...] = (
    BLEMonitorSensorEntityDescription(
        key="temperature",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature",
        unique_id="t_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="cypress temperature",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble cypress temperature",
        unique_id="t_cypress_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature probe 1",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature probe 1",
        unique_id="t_probe_1_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature probe 2",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature probe 2",
        unique_id="t_probe_2_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature probe 3",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature probe 3",
        unique_id="t_probe_3_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature probe 4",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature probe 4",
        unique_id="t_probe_4_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature probe 5",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature probe 5",
        unique_id="t_probe_5_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature probe 6",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature probe 6",
        unique_id="t_probe_6_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature alarm probe 1",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature alarm probe 1",
        unique_id="t_alarm_probe_1_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature alarm probe 2",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature alarm probe 2",
        unique_id="t_alarm_probe_2_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature alarm probe 3",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature alarm probe 3",
        unique_id="t_alarm_probe_3_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature alarm probe 4",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature alarm probe 4",
        unique_id="t_alarm_probe_4_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="low temperature alarm probe 1",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble low temperature alarm probe 1",
        unique_id="t_alarm_low_probe_1_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="low temperature alarm probe 2",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble low temperature alarm probe 2",
        unique_id="t_alarm_low_probe_2_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="low temperature alarm probe 3",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble low temperature alarm probe 3",
        unique_id="t_alarm_low_probe_3_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="low temperature alarm probe 4",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble low temperature alarm probe 4",
        unique_id="t_alarm_low_probe_4_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="temperature calibrated",
        sensor_class="TemperatureSensor",
        update_behavior="Averaging",
        name="ble temperature calibrated",
        unique_id="t_calibrated_",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="humidity",
        sensor_class="HumiditySensor",
        update_behavior="Averaging",
        name="ble humidity",
        unique_id="h_",
        native_unit_of_measurement=PERCENTAGE,
        device_class=SensorDeviceClass.HUMIDITY,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="cypress humidity",
        sensor_class="HumiditySensor",
        update_behavior="Averaging",
        name="ble cypress humidity",
        unique_id="h_cypress_",
        native_unit_of_measurement=PERCENTAGE,
        device_class=SensorDeviceClass.HUMIDITY,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="moisture",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble moisture",
        unique_id="m_",
        native_unit_of_measurement=PERCENTAGE,
        device_class=SensorDeviceClass.HUMIDITY,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="pressure",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble pressure",
        unique_id="p_",
        native_unit_of_measurement=UnitOfPressure.HPA,
        device_class=SensorDeviceClass.PRESSURE,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="water pressure",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble water pressure",
        unique_id="wp_",
        native_unit_of_measurement=UnitOfPressure.BAR,
        device_class=SensorDeviceClass.PRESSURE,
        suggested_display_precision=2,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="conductivity",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble conductivity",
        unique_id="c_",
        icon="mdi:lightning-bolt-circle",
        native_unit_of_measurement=CONDUCTIVITY,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="illuminance",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble illuminance",
        unique_id="l_",
        device_class=SensorDeviceClass.ILLUMINANCE,
        native_unit_of_measurement=LIGHT_LUX,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="formaldehyde",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble formaldehyde",
        unique_id="f_",
        icon="mdi:chemical-weapon",
        native_unit_of_measurement=CONCENTRATION_MILLIGRAMS_PER_CUBIC_METER,
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="dewpoint",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble dewpoint",
        unique_id="d_",
        icon="mdi:water",
        native_unit_of_measurement=UnitOfTemperature.CELSIUS,
        device_class=SensorDeviceClass.TEMPERATURE,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="rssi",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble rssi",
        unique_id="rssi_",
        native_unit_of_measurement=SIGNAL_STRENGTH_DECIBELS_MILLIWATT,
        device_class=SensorDeviceClass.SIGNAL_STRENGTH,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="measured power",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble measured power",
        unique_id="measured_power_",
        native_unit_of_measurement=SIGNAL_STRENGTH_DECIBELS_MILLIWATT,
        device_class=SensorDeviceClass.SIGNAL_STRENGTH,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="battery",
        sensor_class="BatterySensor",
        update_behavior="Averaging",
        name="ble battery",
        unique_id="batt_",
        native_unit_of_measurement=PERCENTAGE,
        device_class=SensorDeviceClass.BATTERY,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="voltage",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble voltage",
        unique_id="v_",
        native_unit_of_measurement=UnitOfElectricPotential.VOLT,
        device_class=SensorDeviceClass.VOLTAGE,
        suggested_display_precision=2,
        state_class=SensorStateClass.MEASUREMENT,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="co2",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="CO2",
        unique_id="co2_",
        icon="mdi:molecule-co2",
        native_unit_of_measurement=CONCENTRATION_PARTS_PER_MILLION,
        device_class=SensorDeviceClass.CO2,
        suggested_display_precision=0,
    ),
    BLEMonitorSensorEntityDescription(
        key="pm2.5",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="PM2.5",
        unique_id="pm25_",
        icon="mdi:molecule",
        native_unit_of_measurement=CONCENTRATION_MICROGRAMS_PER_CUBIC_METER,
        device_class=SensorDeviceClass.PM25,
        suggested_display_precision=0,
    ),
    BLEMonitorSensorEntityDescription(
        key="pm10",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="PM10",
        unique_id="pm10_",
        icon="mdi:molecule",
        native_unit_of_measurement=CONCENTRATION_MICROGRAMS_PER_CUBIC_METER,
        device_class=SensorDeviceClass.PM10,
        suggested_display_precision=0,
    ),
    BLEMonitorSensorEntityDescription(
        key="gravity",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="ble gravity",
        unique_id="gr_",
        icon="mdi:transfer-down",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="tvoc",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="TVOC",
        unique_id="ble_tvoc_",
        icon="mdi:molecule",
        native_unit_of_measurement=CONCENTRATION_MICROGRAMS_PER_CUBIC_METER,
        device_class=SensorDeviceClass.VOLATILE_ORGANIC_COMPOUNDS,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="uv index",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="UV index",
        unique_id="uv_index_",
        icon="mdi:weather-sunny",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="volume",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="volume",
        unique_id="volume_",
        icon="mdi:wave",
        native_unit_of_measurement=UnitOfVolume.LITERS,
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="volume mL",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="volume mL",
        unique_id="volume_mL_",
        icon="mdi:wave",
        native_unit_of_measurement=UnitOfVolume.MILLILITERS,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="volume flow rate",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="volume flow rate",
        unique_id="volume_flow_rate_",
        icon="mdi:wave",
        native_unit_of_measurement="m3/hr",
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="flow",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="flow",
        unique_id="flow_",
        icon="mdi:wave",
        native_unit_of_measurement="l/hr",
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="gas",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="gas",
        unique_id="gas_",
        icon="mdi:gas-burner",
        native_unit_of_measurement=UnitOfVolume.CUBIC_METERS,
        device_class=SensorDeviceClass.GAS,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="water",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="water",
        unique_id="water_",
        icon="mdi:wave",
        native_unit_of_measurement=UnitOfVolume.LITERS,
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="aqi",
        sensor_class="MeasuringSensor",
        update_behavior="Averaging",
        name="Air Quality Index",
        unique_id="ble_iaq_",
        icon="mdi:molecule",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="consumable",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble consumable",
        unique_id="cn_",
        icon="mdi:recycle-variant",
        native_unit_of_measurement=PERCENTAGE,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="opening percentage",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble opening percentage",
        unique_id="op_pct_",
        icon="mdi:garage-open",
        native_unit_of_measurement=PERCENTAGE,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="heart rate",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble heart rate",
        unique_id="hr_",
        icon="mdi:heart-pulse",
        native_unit_of_measurement="bpm",
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="steps",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble steps",
        unique_id="st_",
        icon="mdi:foot-print",
        native_unit_of_measurement="steps",
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.TOTAL,
    ),
    BLEMonitorSensorEntityDescription(
        key="rotation",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble rotation",
        unique_id="rotation_",
        icon="mdi:rotate-left",
        native_unit_of_measurement="°",
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="roll",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="roll",
        unique_id="roll_",
        icon="mdi:horizontal-rotate-clockwise",
        native_unit_of_measurement="°",
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="pitch",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="pitch",
        unique_id="pitch_",
        icon="mdi:rotate-right-variant",
        native_unit_of_measurement="°",
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="distance",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="distance",
        unique_id="distance_",
        icon="mdi:map-marker-distance",
        native_unit_of_measurement="m",
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="distance mm",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="distance mm",
        unique_id="distance_mm_",
        icon="mdi:map-marker-distance",
        native_unit_of_measurement="mm",
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="duration",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="duration",
        unique_id="duration_",
        icon="mdi:clock-time-eight",
        native_unit_of_measurement="s",
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="current",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="current",
        unique_id="current_",
        icon="mdi:sine-wave",
        native_unit_of_measurement="A",
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="speed",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="speed",
        unique_id="speed_",
        icon="mdi:speedometer",
        native_unit_of_measurement="m/s",
        device_class=None,
        suggested_display_precision=2,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="pulse",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble pulse",
        unique_id="pu_",
        icon="mdi:heart-pulse",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="shake",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble shake",
        unique_id="sh_",
        icon="mdi:vibrate",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=0,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="gyroscope",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="gyroscope",
        unique_id="gyro_",
        icon="mdi:rotate-orbit",
        native_unit_of_measurement="°/s",
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="impedance",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble impedance",
        unique_id="im_",
        icon="mdi:omega",
        native_unit_of_measurement="Ohm",
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="magnetic field",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble magnetic field",
        unique_id="mf_",
        icon="mdi:magnet",
        native_unit_of_measurement="µT",
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="magnetic field direction",
        sensor_class="InstantUpdateSensor",
        update_behavior="Instantly",
        name="ble magnetic field direction",
        unique_id="mfd_",
        icon="mdi:compass",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="mac",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="ble mac",
        unique_id="mac_",
        icon="mdi:alpha-m-circle-outline",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="uuid",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="ble uuid",
        unique_id="uuid_",
        icon="mdi:alpha-u-circle-outline",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="major",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="ble major",
        unique_id="major_",
        icon="mdi:counter",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="minor",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="ble minor",
        unique_id="minor_",
        icon="mdi:counter",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    BLEMonitorSensorEntityDescription(
        key="count",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="ble count",
        unique_id="cnt_",
        icon="mdi:counter",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="movement counter",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="movement counter",
        unique_id="mv_cnt_",
        icon="mdi:counter",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="score",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="ble score",
        unique_id="sco_",
        icon="mdi:counter",
        native_unit_of_measurement=None,
        device_class=None,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="air quality",
        sensor_class="StateChangedSensor",
        update_behavior="StateChange",
        name="Air Quality",
        unique_id="ble_aq_",
        icon="mdi:molecule",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="text",
        sensor_class="StateChangedSensor",
        update_behavior="Instantly",
        name="text",
        unique_id="text_",
        icon="mdi:cursor-text",
        native_unit_of_measurement=None,
        device_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="pump mode",
        sensor_class="StateChangedSensor",
        update_behavior="Instantly",
        name="pump mode",
        unique_id="pump_mode_",
        icon="mdi:pump",
        native_unit_of_measurement=None,
        device_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="timestamp",
        sensor_class="StateChangedSensor",
        update_behavior="Instantly",
        name="timestamp",
        unique_id="timestamp_",
        device_class=SensorDeviceClass.TIMESTAMP,
    ),
    BLEMonitorSensorEntityDescription(
        key="acceleration",
        sensor_class="AccelerationSensor",
        update_behavior="Instantly",
        name="ble acceleration",
        unique_id="ac_",
        icon="mdi:axis-arrow",
        native_unit_of_measurement="mG",
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="weight",
        sensor_class="WeightSensor",
        update_behavior="Instantly",
        name="ble weight",
        unique_id="w_",
        icon="mdi:scale-bathroom",
        native_unit_of_measurement=UnitOfMass.KILOGRAMS,
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="stabilized weight",
        sensor_class="WeightSensor",
        update_behavior="Instantly",
        name="ble stabilized weight",
        unique_id="stab_w_",
        icon="mdi:scale-bathroom",
        native_unit_of_measurement=UnitOfMass.KILOGRAMS,
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="non-stabilized weight",
        sensor_class="WeightSensor",
        update_behavior="Instantly",
        name="ble non-stabilized weight",
        unique_id="nw_",
        icon="mdi:scale-bathroom",
        native_unit_of_measurement=UnitOfMass.KILOGRAMS,
        device_class=None,
        suggested_display_precision=1,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="energy",
        sensor_class="EnergySensor",
        update_behavior="Instantly",
        name="ble energy",
        unique_id="e_",
        native_unit_of_measurement=UnitOfEnergy.KILO_WATT_HOUR,
        device_class=SensorDeviceClass.ENERGY,
        suggested_display_precision=2,
        state_class=SensorStateClass.TOTAL_INCREASING,
    ),
    BLEMonitorSensorEntityDescription(
        key="power",
        sensor_class="PowerSensor",
        update_behavior="Instantly",
        name="ble power",
        unique_id="pow_",
        native_unit_of_measurement=UnitOfPower.KILO_WATT,
        device_class=SensorDeviceClass.POWER,
        suggested_display_precision=0,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="button",
        sensor_class="ButtonSensor",
        update_behavior="Instantly",
        name="ble button",
        unique_id="bu_",
        icon="mdi:gesture-tap-button",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="dimmer",
        sensor_class="DimmerSensor",
        update_behavior="Instantly",
        name="ble dimmer",
        unique_id="di_",
        icon="mdi:rotate-right",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="one btn switch",
        sensor_class="SwitchSensor",
        update_behavior="Instantly",
        name="ble one button switch",
        unique_id="switch_",
        icon="mdi:gesture-tap-button",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="two btn switch left",
        sensor_class="SwitchSensor",
        update_behavior="Instantly",
        name="ble two button switch left",
        unique_id="left_switch_",
        icon="mdi:gesture-tap-button",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="two btn switch right",
        sensor_class="SwitchSensor",
        update_behavior="Instantly",
        name="ble two button switch right",
        unique_id="right_switch_",
        icon="mdi:gesture-tap-button",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="three btn switch left",
        sensor_class="SwitchSensor",
        update_behavior="Instantly",
        name="ble three button switch left",
        unique_id="left_switch_",
        icon="mdi:gesture-tap-button",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="three btn switch middle",
        sensor_class="SwitchSensor",
        update_behavior="Instantly",
        name="ble three button switch middle",
        unique_id="middle_switch_",
        icon="mdi:gesture-tap-button",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="three btn switch right",
        sensor_class="SwitchSensor",
        update_behavior="Instantly",
        name="ble three button switch right",
        unique_id="right_switch_",
        icon="mdi:gesture-tap-button",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="remote",
        sensor_class="BaseRemoteSensor",
        update_behavior="Instantly",
        name="ble remote",
        unique_id="re_",
        icon="mdi:remote",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="fan remote",
        sensor_class="BaseRemoteSensor",
        update_behavior="Instantly",
        name="ble fan remote",
        unique_id="fa_",
        icon="mdi:remote",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="ventilator fan remote",
        sensor_class="BaseRemoteSensor",
        update_behavior="Instantly",
        name="ble ventilator fan remote",
        unique_id="fr_",
        icon="mdi:remote",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="bathroom heater remote",
        sensor_class="BaseRemoteSensor",
        update_behavior="Instantly",
        name="ble bathroom heater remote",
        unique_id="bh_",
        icon="mdi:remote",
        native_unit_of_measurement=None,
        device_class=None,
        state_class=None,
    ),
    BLEMonitorSensorEntityDescription(
        key="volume dispensed port 1",
        sensor_class="VolumeDispensedSensor",
        update_behavior="Instantly",
        name="ble volume dispensed port 1",
        unique_id="vd1_",
        icon="mdi:keg",
        native_unit_of_measurement=UnitOfVolume.LITERS,
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
    BLEMonitorSensorEntityDescription(
        key="volume dispensed port 2",
        sensor_class="VolumeDispensedSensor",
        update_behavior="Instantly",
        name="ble volume dispensed port 2",
        unique_id="vd2_",
        icon="mdi:keg",
        native_unit_of_measurement=UnitOfVolume.LITERS,
        device_class=None,
        suggested_display_precision=3,
        state_class=SensorStateClass.MEASUREMENT,
    ),
)


# Dictionary with supported sensors
# Format {device: [averaging sensor list], [instantly updating sensor list],[binary sensor list]}:
# - [averaging sensor list]:            sensors that update the state after avering of the data
# - [instantly updating sensor list]:   sensors that update the state instantly after new data
# - [binary sensor list]:               binary sensors
MEASUREMENT_DICT = {
    'LYWSDCGQ'                : [["temperature", "humidity", "battery", "rssi"], [], []],
    'LYWSD02'                 : [["temperature", "humidity", "battery", "rssi"], [], []],
    'LYWSD02MMC'              : [["temperature", "humidity", "battery", "rssi"], [], []],
    'LYWSD03MMC'              : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'XMWSDJ04MMC'             : [["temperature", "humidity", "battery", "rssi"], [], []],
    'XMMF01JQD'               : [["rssi"], ["button"], []],
    'HHCCJCY01'               : [["temperature", "moisture", "conductivity", "illuminance", "battery", "rssi"], [], []],
    'GCLS002'                 : [["temperature", "moisture", "conductivity", "illuminance", "rssi"], [], []],
    'HHCCPOT002'              : [["moisture", "conductivity", "rssi"], [], []],
    'WX08ZM'                  : [["consumable", "battery", "rssi"], [], ["switch"]],
    'MCCGQ02HL'               : [["battery", "rssi"], [], ["opening", "light"]],
    'YM-K1501'                : [["rssi"], ["temperature"], ["switch"]],
    'YM-K1501EU'              : [["rssi"], ["temperature"], ["switch"]],
    'V-SK152'                 : [["rssi"], ["temperature"], ["switch"]],
    'SJWS01LM'                : [["battery", "rssi"], ["button"], ["moisture detected"]],
    'MJYD02YL'                : [["battery", "rssi"], [], ["light", "motion"]],
    'MUE4094RT'               : [["rssi"], [], ["motion"]],
    'RTCGQ02LM'               : [["battery", "rssi"], ["button"], ["light", "motion"]],
    'MMC-T201-1'              : [["temperature", "battery", "rssi"], [], []],
    'M1S-T500'                : [["score", "battery", "rssi"], [], ["toothbrush"]],
    'T700'                    : [["consumable", "score", "battery", "rssi"], [], ["toothbrush"]],
    'T700i'                   : [["consumable", "score", "battery", "rssi"], [], ["toothbrush"]],
    'ZNMS16LM'                : [["battery", "rssi"], [], ["lock", "door", "fingerprint", "armed away"]],
    'ZNMS17LM'                : [["battery", "rssi"], [], ["lock", "door", "fingerprint", "antilock", "childlock", "armed away"]],
    'SV40'                    : [["battery", "rssi"], [], ["lock", "door", "fingerprint", "antilock", "childlock", "armed away"]],
    'MJZNMSQ01YD'             : [["battery", "rssi"], [], ["lock", "fingerprint"]],
    'MJWSD05MMC'              : [["temperature", "humidity", "battery", "rssi"], [], []],
    'XMZNMST02YD'             : [["battery", "rssi"], [], ["lock", "door", "fingerprint"]],
    'XMZNMS04LM'              : [["battery", "rssi"], [], ["lock", "fingerprint"]],
    'XMZNMS08LM'              : [["battery", "rssi"], [], ["lock", "door"]],
    'MJZNZ018H'               : [["battery", "rssi"], ["button"], ["bed occupancy", "snoring", "sleeping"]],
    'ZX1'                     : [["rssi"], ["button"], ["bed occupancy", "snoring", "sleeping"]],
    'CGC1'                    : [["temperature", "humidity", "battery", "rssi"], [], []],
    'CGD1'                    : [["temperature", "humidity", "battery", "rssi"], [], []],
    'CGDK2'                   : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'CGG1'                    : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'CGG1-ENCRYPTED'          : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'CGH1'                    : [["battery", "rssi"], [], ["opening"]],
    'CGP1W'                   : [["temperature", "humidity", "battery", "pressure", "rssi"], [], []],
    'CGPR1'                   : [["illuminance", "battery", "rssi"], [], ["light", "motion"]],
    'CGDN1'                   : [["temperature", "humidity", "co2", "pm2.5", "pm10", "rssi"], [], []],
    'MHO-C401'                : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'MHO-C303'                : [["temperature", "humidity", "battery", "rssi"], [], []],
    'JQJCY01YM'               : [["temperature", "humidity", "battery", "formaldehyde", "rssi"], [], []],
    'JTYJGD03MI'              : [["rssi"], ["button", "battery"], ["smoke detector"]],
    'K9B-1BTN'                : [["rssi"], ["one btn switch"], []],
    'K9B-2BTN'                : [["rssi"], ["two btn switch left", "two btn switch right"], []],
    'K9B-3BTN'                : [["rssi"], ["three btn switch left", "three btn switch middle", "three btn switch right"], []],
    'K9BB-1BTN'               : [["battery", "rssi"], ["one btn switch"], []],
    'MS1BB(MI)'               : [["battery", "rssi"], ["button"], ["opening"]],
    'HS1BB(MI)'               : [["illuminance", "battery", "rssi"], [], ["motion"]],
    'XMWXKG01YL'              : [["rssi"], ["two btn switch left", "two btn switch right"], []],
    'XMWXKG01LM'              : [["battery", "rssi"], ["one btn switch"], []],
    'YLAI003'                 : [["rssi", "battery"], ["button"], []],
    'YLYK01YL'                : [["rssi"], ["remote"], ["remote single press", "remote long press"]],
    'YLYK01YL-FANCL'          : [["rssi"], ["fan remote"], []],
    'YLYK01YL-VENFAN'         : [["rssi"], ["ventilator fan remote"], []],
    'YLYB01YL-BHFRC'          : [["rssi"], ["bathroom heater remote"], []],
    'YLKG07YL/YLKG08YL'       : [["rssi"], ["dimmer"], []],
    'SU001-T'                 : [["battery", "rssi"], [], ["motion", "switch"]],
    'ATC'                     : [["temperature", "humidity", "battery", "voltage", "rssi"], [], ["switch", "opening"]],
    'Mi Scale V1'             : [["rssi"], ["weight", "non-stabilized weight"], ["weight removed"]],
    'Mi Scale V2'             : [["rssi"], ["weight", "stabilized weight", "non-stabilized weight", "impedance"], ["weight removed"]],
    'Mi Band'                 : [["rssi", "heart rate", "steps"], [], []],
    'TZC4'                    : [["rssi"], ["weight", "non-stabilized weight", "impedance"], []],
    'QJ-J'                    : [["rssi"], ["weight", "non-stabilized weight", "impedance"], []],
    'Kegtron KT-100'          : [["rssi"], ["volume dispensed port 1"], []],
    'Kegtron KT-200'          : [["rssi"], ["volume dispensed port 1", "volume dispensed port 2"], []],
    'Smart hygrometer'        : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'Lanyard/mini hygrometer' : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'T201'                    : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    'H5072/H5075'             : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5101/H5102/H5177'       : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5051/H5071'             : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5052'                   : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5071'                   : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5074'                   : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5106'                   : [["temperature", "humidity", "pm2.5", "rssi"], [], []],
    'H5178'                   : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5178-outdoor'           : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5179'                   : [["temperature", "humidity", "battery", "rssi"], [], []],
    'H5182'                   : [["temperature probe 1", "temperature alarm probe 1", "temperature probe 2", "temperature alarm probe 2", "rssi"], [], []],
    'H5183'                   : [["temperature probe 1", "temperature alarm probe 1", "rssi"], [], []],
    'H5184'                   : [["temperature probe 1", "temperature alarm probe 1", "temperature probe 2", "temperature alarm probe 2", "temperature probe 3", "temperature alarm probe 3", "temperature probe 4", "temperature alarm probe 4", "rssi"], [], []],
    'H5185'                   : [["temperature probe 1", "temperature alarm probe 1", "temperature probe 2", "temperature alarm probe 2", "rssi"], [], []],
    'H5198'                   : [["temperature probe 1", "temperature alarm probe 1", "low temperature alarm probe 1", "temperature probe 2", "temperature alarm probe 2", "low temperature alarm probe 2", "temperature probe 3", "temperature alarm probe 3", "low temperature alarm probe 3", "temperature probe 4", "temperature alarm probe 4", "low temperature alarm probe 4", "rssi"], [], []],
    'Ruuvitag'                : [["temperature", "humidity", "pressure", "battery", "voltage", "rssi"], ["acceleration"], ["motion"]],
    'iNode Energy Meter'      : [["battery", "voltage", "rssi"], ["energy", "power"], []],
    "iNode Care Sensor 1"     : [["temperature", "battery", "voltage", "rssi"], ["acceleration"], ["motion"]],
    "iNode Care Sensor 2"     : [["temperature", "battery", "voltage", "rssi"], ["acceleration"], ["motion"]],
    "iNode Care Sensor 3"     : [["temperature", "humidity", "battery", "voltage", "rssi"], ["acceleration"], ["motion"]],
    "iNode Care Sensor 4"     : [["temperature", "battery", "voltage", "rssi"], ["acceleration"], ["motion"]],
    "iNode Care Sensor 5"     : [["temperature", "battery", "voltage", "rssi"], ["acceleration", "magnetic field", "magnetic field direction"], ["motion"]],
    "iNode Care Sensor 6"     : [["temperature", "battery", "voltage", "rssi"], ["acceleration"], ["motion"]],
    "iNode Care Sensor T"     : [["temperature", "battery", "voltage", "rssi"], [], []],
    "iNode Care Sensor HT"    : [["temperature", "humidity", "battery", "voltage", "rssi"], [], []],
    "iNode Care Sensor PT"    : [["temperature", "pressure", "battery", "voltage", "rssi"], [], []],
    "iNode Care Sensor PHT"   : [["temperature", "humidity", "pressure", "battery", "voltage", "rssi"], [], []],
    'HTP.xw'                  : [["temperature", "humidity", "pressure", "rssi"], [], []],
    'HT.w'                    : [["temperature", "humidity", "pressure", "rssi"], [], []],
    'MyCO2'                   : [["temperature", "humidity", "co2", "rssi"], [], []],
    'SHT40 Gadget'            : [["temperature", "humidity", "rssi"], [], []],
    'SHT41 Gadget'            : [["temperature", "humidity", "rssi"], [], []],
    'SHT45 Gadget'            : [["temperature", "humidity", "rssi"], [], []],
    'Moat S2'                 : [["temperature", "humidity", "battery", "rssi"], [], []],
    'Pebble'                  : [["temperature", "humidity", "pressure", "rssi"], [], []],
    'Tempo Disc THD'          : [["temperature", "humidity", "dewpoint", "battery", "rssi"], [], []],
    'Tempo Disc THPD'         : [["temperature", "humidity", "pressure", "battery", "rssi"], [], []],
    'b-parasite V1.0.0'       : [["temperature", "humidity", "moisture", "voltage", "rssi"], [], []],
    'b-parasite V1.1.0'       : [["temperature", "humidity", "moisture", "voltage", "rssi", "illuminance"], [], []],
    'SmartSeries 7000'        : [["rssi"], [], ["toothbrush"]],
    'IO Series 7'             : [["rssi"], [], ["toothbrush"]],
    'iBBQ-1'                  : [["temperature probe 1", "rssi"], [], []],
    'iBBQ-2'                  : [["temperature probe 1", "temperature probe 2", "rssi"], [], []],
    'iBBQ-4'                  : [["temperature probe 1", "temperature probe 2", "temperature probe 3", "temperature probe 4", "rssi"], [], []],
    'iBBQ-6'                  : [["temperature probe 1", "temperature probe 2", "temperature probe 3", "temperature probe 4", "temperature probe 5", "temperature probe 6", "rssi"], [], []],
    'BEC07-5'                 : [["temperature", "humidity", "rssi"], [], []],
    'iBeacon'                 : [["rssi", "measured power", "cypress temperature", "cypress humidity"], ["uuid", "mac", "major", "minor"], []],  # mac can be dynamic
    'AltBeacon'               : [["rssi", "measured power"], ["uuid", "mac", "major", "minor"], []],  # mac can be dynamic
    'Air Mentor Pro 2'        : [["temperature", "temperature calibrated", "humidity", "co2", "tvoc", "aqi", "air quality", "pm2.5", "pm10", "rssi"], [], []],
    'Air Mentor 2S'           : [["temperature", "temperature calibrated", "humidity", "co2", "tvoc", "aqi", "formaldehyde", "air quality", "pm2.5", "pm10", "rssi"], [], []],
    'bluSensor Mini'          : [["temperature", "humidity", "co2", "tvoc", "aqi", "rssi"], [], []],
    'Meter TH S1'             : [["temperature", "humidity", "battery", "rssi"], [], []],
    'Meter TH plus'           : [["temperature", "humidity", "battery", "rssi"], [], []],
    'Laica Smart Scale'       : [["weight", "impedance", "rssi"], [], []],
    "Acconeer XM122"          : [["temperature", "battery", "rssi"], [], ["motion"]],
    "Acconeer XM126"          : [["temperature", "battery", "rssi"], [], ["motion"]],
    'K6 Sensor Beacon'        : [["temperature", "humidity", "acceleration", "voltage", "battery", "rssi"], [], []],
    'DSL-C08'                 : [["battery", "rssi", "voltage"], [], ["lock", "childlock"]],
    'SmartDry cloth dryer'    : [["temperature", "humidity", "voltage", "battery", "shake", "rssi"], [], ["switch"]],
}

# Sensor manufacturer dictionary
MANUFACTURER_DICT = {
    'DSL-C08'                 : 'Lockin',
    'LYWSDCGQ'                : 'Xiaomi',
    'LYWSD02'                 : 'Xiaomi',
    'LYWSD02MMC'              : 'Xiaomi',
    'LYWSD03MMC'              : 'Xiaomi',
    'XMWSDJ04MMC'             : 'Xiaomi',
    'XMMF01JQD'               : 'Xiaomi',
    'HHCCJCY01'               : 'Xiaomi',
    'GCLS002'                 : 'Xiaomi',
    'HHCCPOT002'              : 'Xiaomi',
    'WX08ZM'                  : 'Xiaomi',
    'MCCGQ02HL'               : 'Xiaomi',
    'YM-K1501'                : 'Xiaomi',
    'YM-K1501EU'              : 'Xiaomi',
    'V-SK152'                 : 'Viomi',
    'SJWS01LM'                : 'Xiaomi',
    'MJYD02YL'                : 'Xiaomi',
    'MUE4094RT'               : 'Xiaomi',
    'RTCGQ02LM'               : 'Xiaomi',
    'MMC-T201-1'              : 'Xiaomi',
    'T700'                    : 'Xiaomi',
    'T700i'                   : 'Xiaomi',
    'M1S-T500'                : 'Xiaomi Soocas',
    'ZNMS16LM'                : 'Xiaomi Aqara',
    'ZNMS17LM'                : 'Xiaomi Aqara',
    'MJZNMSQ01YD'             : 'Xiaomi',
    'MJWSD05MMC'              : 'Xiaomi',
    'XMZNMST02YD'             : 'Xiaomi',
    'XMZNMS04LM'              : 'Xiaomi',
    'XMZNMS08LM'              : 'Xiaomi',
    'MJZNZ018H'               : 'Xiaomi',
    'ZX1'                     : '8H Sleep',
    'CGC1'                    : 'Qingping',
    'CGD1'                    : 'Qingping',
    'CGDK2'                   : 'Qingping',
    'CGG1'                    : 'Qingping',
    'CGG1-ENCRYPTED'          : 'Qingping',
    'CGH1'                    : 'Qingping',
    'CGP1W'                   : 'Qingping',
    'CGPR1'                   : 'Qingping',
    'CGDN1'                   : 'Qingping',
    'MHO-C401'                : 'Miaomiaoce',
    'MHO-C303'                : 'Miaomiaoce',
    'JQJCY01YM'               : 'Honeywell',
    'JTYJGD03MI'              : 'Honeywell',
    'YLAI003'                 : 'Yeelight',
    'YLYK01YL'                : 'Yeelight',
    'YLYK01YL-FANCL'          : 'Yeelight',
    'YLYK01YL-VENFAN'         : 'Yeelight',
    'YLYB01YL-BHFRC'          : 'Yeelight',
    'YLKG07YL/YLKG08YL'       : 'Yeelight',
    'K9B-1BTN'                : 'Linptech',
    'K9B-2BTN'                : 'Linptech',
    'K9B-3BTN'                : 'Linptech',
    'K9BB-1BTN'               : 'Linptech',
    'MS1BB(MI)'               : 'Linptech',
    'HS1BB(MI)'               : 'Linptech',
    'XMWXKG01YL'              : 'Xiaomi',
    'XMWXKG01LM'              : 'Xiaomi',
    'SV40'                    : 'Lockin',
    'SU001-T'                 : 'Petoneer',
    'ATC'                     : 'ATC',
    'Mi Scale V1'             : 'Xiaomi',
    'Mi Scale V2'             : 'Xiaomi',
    'Mi Band'                 : 'Xiaomi',
    'TZC4'                    : 'Xiaogui',
    'QJ-J'                    : 'MaxxMee',
    'Kegtron KT-100'          : 'Kegtron',
    'Kegtron KT-200'          : 'Kegtron',
    'Smart hygrometer'        : 'Thermobeacon',
    'Lanyard/mini hygrometer' : 'Thermobeacon',
    'T201'                    : 'Thermobeacon',
    'H5072/H5075'             : 'Govee',
    'H5101/H5102/H5177'       : 'Govee',
    'H5051/H5071'             : 'Govee',
    'H5052'                   : 'Govee',
    'H5071'                   : 'Govee',
    'H5074'                   : 'Govee',
    'H5106'                   : 'Govee',
    'H5178'                   : 'Govee',
    'H5178-outdoor'           : 'Govee',
    'H5179'                   : 'Govee',
    'H5182'                   : 'Govee',
    'H5183'                   : 'Govee',
    'H5184'                   : 'Govee',
    'H5185'                   : 'Govee',
    'H5198'                   : 'Govee',
    'Ruuvitag'                : 'Ruuvitag',
    'iNode Energy Meter'      : 'iNode',
    "iNode Care Sensor 1"     : 'iNode',
    "iNode Care Sensor 2"     : 'iNode',
    "iNode Care Sensor 3"     : 'iNode',
    "iNode Care Sensor 4"     : 'iNode',
    "iNode Care Sensor 5"     : 'iNode',
    "iNode Care Sensor 6"     : 'iNode',
    "iNode Care Sensor T"     : 'iNode',
    "iNode Care Sensor HT"    : 'iNode',
    "iNode Care Sensor PT"    : 'iNode',
    "iNode Care Sensor PHT"   : 'iNode',
    'HTP.xw'                  : 'SensorPush',
    'HT.w'                    : 'SensorPush',
    'MyCO2'                   : 'Sensirion',
    'SHT40 Gadget'            : 'Sensirion',
    'SHT41 Gadget'            : 'Sensirion',
    'SHT45 Gadget'            : 'Sensirion',
    'Moat S2'                 : 'Moat',
    'Pebble'                  : 'BlueMaestro',
    'Tempo Disc THD'          : 'BlueMaestro',
    'Tempo Disc THPD'         : 'BlueMaestro',
    'b-parasite V1.0.0'       : 'rbaron',
    'b-parasite V1.1.0'       : 'rbaron',
    'SmartSeries 7000'        : 'Oral-B',
    'IO Series 7'             : 'Oral-B',
    'iBBQ-1'                  : 'Inkbird',
    'iBBQ-2'                  : 'Inkbird',
    'iBBQ-4'                  : 'Inkbird',
    'iBBQ-6'                  : 'Inkbird',
    'BEC07-5'                 : 'Jinou',
    'iBeacon'                 : 'Apple',
    'AltBeacon'               : 'Radius Networks',
    'EClerk Eco'              : 'Relsib',
    'Air Mentor Pro 2'        : 'Air Mentor',
    'Air Mentor 2S'           : 'Air Mentor',
    'bluSensor Mini'          : 'Almendo',
    'Meter TH S1'             : 'Switchbot',
    'Meter TH plus'           : 'Switchbot',
    'Laica Smart Scale'       : 'Laica',
    'Acconeer XM122'          : 'Acconeer',
    'Acconeer XM126'          : 'Acconeer',
    'K6 Sensor Beacon'        : 'KKM',
    'SmartDry cloth dryer'    : 'SmartDry',
}


# Renamed model dictionary
RENAMED_MODEL_DICT = {
    'HA BLE DIY': 'BTHome',
    'LINP-M1': 'MS1BB(MI)',
    'M1SBB(MI)': 'MS1BB(MI)',
    'H5051/H5071': 'H5051',
}


# Renamed firmware dictionary
RENAMED_FIRMWARE_DICT = {
    'HA BLE': 'BTHome',
    'HA BLE (encrypted)': 'BTHome (encrypted)',
    'Thermoplus': 'Thermobeacon',
    'Brifit': 'Thermobeacon',
    'Relsib (EClerk Eco v9a)': 'Relsib',
}


# Renamed manufacturer dictionary
RENAMED_MANUFACTURER_DICT = {
    'Home Assistant DIY': 'BTHome',
    'Thermoplus': 'Thermobeacon',
    'Brifit': 'Thermobeacon',
}


# Sensors that support automatic adding of sensors and binary sensors
AUTO_MANUFACTURER_DICT = {
    'Amazfit Smart Scale'     : 'Amazfit',
    'Blustream'               : 'Blustream',
    'BTHome'                  : 'BTHome',
    'MI401'                   : 'Grundfos',
    'HHCCJCY10'               : 'HHCC',
    'HolyIOT BLE tracker'     : 'HolyIOT',
    'Supramatic E4 BS'        : 'Hörmann',
    'IBS-TH'                  : 'Inkbird',
    'IBS-TH2/P01B'            : 'Inkbird',
    'JHT'                     : 'Jaalee',
    'TG-BT5-IN'               : 'Mikrotik',
    'TG-BT5-OUT'              : 'Mikrotik',
    'Electra Washbasin Faucet': 'Oras',
    'CGP23W'                  : 'Qingping',
    'EClerk Eco'              : 'Relsib',
    'WT51'                    : 'Relsib',
    'Blue Puck T'             : 'Teltonika',
    'Blue Coin T'             : 'Teltonika',
    'Blue Puck RHT'           : 'Teltonika',
    'EYE sensor'              : 'Teltonika',
    'TP357'                   : 'Thermopro',
    'TP359'                   : 'Thermopro',
    'Tilt Red'                : 'Tilt',
    'Tilt Green'              : 'Tilt',
    'Tilt Black'              : 'Tilt',
    'Tilt Purple'             : 'Tilt',
    'Tilt Orange'             : 'Tilt',
    'Tilt Blue'               : 'Tilt',
    'Tilt Yellow'             : 'Tilt',
    'Tilt Pink'               : 'Tilt',
    'MMC-W505'                : 'Xiaomi',
}


# Binary Sensors that are automatically added if device is in AUTO_MANUFACTURER_DICT
AUTO_BINARY_SENSOR_LIST = [
    "battery charging",
    "battery low",
    "carbon monoxide",
    "cold",
    "connectivity",
    "binary",
    "door",
    "dropping",
    "garage door",
    "gas detected",
    "heat",
    "impact",
    "light",
    "lock",
    "magnetic field detected",
    "motion",
    "moisture detected",
    "moving",
    "occupancy",
    "opening",
    "plug",
    "presence",
    "problem",
    "running",
    "safety",
    "smoke",
    "sound",
    "switch",
    "sound",
    "tamper",
    "tilt",
    "vibration",
    "window",
]


# Sensors that are automatically added if device is in AUTO_MANUFACTURER_DICT
AUTO_SENSOR_LIST = [
    "acceleration",
    "battery",
    "button",
    "co2",
    "conductivity",
    "count",
    "current",
    "dewpoint",
    "dimmer",
    "distance",
    "distance mm",
    "duration",
    "energy",
    "flow",
    "gas",
    "gravity",
    "gyroscope",
    "humidity",
    "illuminance",
    "impedance",
    "moisture",
    "movement counter",
    "non-stabilized weight",
    "opening percentage",
    "pitch",
    "pm2.5",
    "pm10",
    "power",
    "pressure",
    "pulse",
    "pump mode",
    "roll",
    "rotation",
    "speed",
    "steps",
    "rssi",
    "temperature",
    "temperature probe 1",
    "text",
    "timestamp",
    "tvoc",
    "uv index",
    "voltage",
    "volume",
    "volume mL",
    "volume flow rate",
    "water",
    "water pressure",
    "weight",
]


# Selection list for report_uknown
REPORT_UNKNOWN_LIST = [
    "Off",
    "Acconeer",
    "Air Mentor",
    "Amazfit",
    "ATC",
    "BlueMaestro",
    "Blustream",
    "BTHome",
    "Govee",
    "Grundfos",
    "HHCC",
    'HolyIOT',
    "Hormann",
    "Inkbird",
    "iNode",
    "iBeacon",
    "Jaalee",
    "Jinou",
    "Kegtron",
    "KKM",
    "Mi Band",
    "Mi Scale",
    "Mikrotik",
    "Moat",
    "Oras",
    "Oral-B",
    "Qingping",
    "Relsib",
    "rbaron",
    "Ruuvitag",
    "Sensirion",
    "SensorPush",
    "SmartDry",
    "Switchbot",
    "Teltonika",
    "Thermobeacon",
    "Tilt",
    "Xiaogui",
    "Xiaomi",
    "Other",
    False,
]
