"""Tuya Energy Meter."""

from typing import Optional, Union
from zigpy.profiles import zgp, zha
import zigpy.types as t
from zigpy.zcl.clusters.general import Basic, GreenPowerProxy, Groups, Ota, Scenes, Time
from zhaquirks.const import (
    DEVICE_TYPE,
    ENDPOINTS,
    INPUT_CLUSTERS,
    MODELS_INFO,
    OUTPUT_CLUSTERS,
    PROFILE_ID,
)
from zhaquirks.tuya import (
    AttributeWithMask,
    NoManufacturerCluster,
    TuyaDatapointData,
    TuyaLocalCluster,
    TuyaManufClusterAttributes,
    TuyaZBElectricalMeasurement,
    TuyaZBMeteringClusterWithUnit,
)
from zhaquirks.tuya.mcu import (
    DPToAttributeMapping,
    EnchantedDevice,
    TuyaMCUCluster,
)

class TuyaElectricalMeasurement(TuyaZBElectricalMeasurement, TuyaLocalCluster):
    """Tuya local ElectricalMeasurement cluster."""

class TuyaMetering(TuyaZBMeteringClusterWithUnit, TuyaLocalCluster):
    """Tuya local Metering cluster."""

    METERING_DEVICE_TYPE = 0x0306
    STATUS = 0x0200
    SUMMATION_FORMATTING = 0x0303

    """Define formatting and mandatory attributes."""
    _CONSTANT_ATTRIBUTES = {
        **TuyaZBMeteringClusterWithUnit._CONSTANT_ATTRIBUTES,
        METERING_DEVICE_TYPE: 0x00,
        STATUS: 0x00,
        SUMMATION_FORMATTING: 0b01111010,
    }

class TuyaMatSeePlusElectricalMeasurement(TuyaElectricalMeasurement):
    """Tuya MatSee ElectricalMeasurement cluster."""

    AC_FREQUENCY_DIVISOR = 0x0401
    AC_FREQUENCY_MULTIPLIER = 0x0400
    AC_POWER_DIVISOR = 0x0605
    AC_POWER_MULTIPLIER = 0x0604
    AC_VOLTAGE_DIVISOR = 0x0601
    AC_VOLTAGE_MULTIPLIER = 0x0600

    """Define formatting and mandatory attributes."""
    _CONSTANT_ATTRIBUTES = {
        **TuyaElectricalMeasurement._CONSTANT_ATTRIBUTES,
        AC_FREQUENCY_DIVISOR: 100,
        AC_FREQUENCY_MULTIPLIER: 1,
        AC_POWER_DIVISOR: 1,
        AC_POWER_MULTIPLIER: 1,
        AC_VOLTAGE_DIVISOR: 10,
        AC_VOLTAGE_MULTIPLIER: 1,
    }

    """MatSee Plus specific attribute for signing uint32 active power values."""
    attributes = {
        **TuyaElectricalMeasurement.attributes,
        0x0f01: ("power_flow", t.uint8_t, True),
    }

    def get_cluster_attribute(self, ep_attribute: str, attr_name: str, endpoint_id: int, default=None):
        """Get the value of an endpoint's cluster attribute."""

        if endpoint_id:
            endpoint = self.endpoint.device.endpoints[endpoint_id]
        else:
            endpoint = self.endpoint
        cluster = getattr(endpoint, ep_attribute)
        return cluster.get(attr_name, default)

    def update_attribute(self, attr_name: str, value) -> None:
        """Manufacturer specific attribute handling"""

        """Convert device reported uint32 to signed int32 using reported power_flow."""
        if attr_name == "active_power":
            power_flow = self.get("power_flow")
            if power_flow in (0,1):
                value = int(value * (-1 if power_flow else 1))
            elif power_flow != None:
                return

        super().update_attribute(attr_name, value)

        """Calculate reactive power."""
        if attr_name == "active_power":
            apparent_power = self.get("apparent_power")
            if apparent_power != None:
                reactive_power = (apparent_power**2 - value**2) ** 0.5
                super().update_attribute("reactive_power", int(reactive_power))

        """Calculate apparent power."""
        if attr_name == "rms_current":
            rms_voltage = self.get_cluster_attribute(self.ep_attribute, "rms_voltage", 1)
            if rms_voltage != None:
                ac_current_divisor = self.get("ac_current_divisor", 1)
                ac_current_multiplier = self.get("ac_current_multiplier", 1)
                ac_power_divisor = self.get("ac_power_divisor", 1)
                ac_power_multiplier = self.get("ac_power_multiplier", 1)
                ac_voltage_divisor = self.get("ac_voltage_divisor", 1)
                ac_voltage_multiplier = self.get("ac_voltage_multiplier", 1)
                
                current = value / ac_current_divisor * ac_current_multiplier*1.13
                voltage = rms_voltage / ac_voltage_divisor * ac_voltage_multiplier
                apparent_power = (current * voltage) * ac_power_divisor / ac_power_multiplier
                super().update_attribute("apparent_power", int(apparent_power))


class TuyaMatSeePlusManufCluster(NoManufacturerCluster, TuyaMCUCluster, TuyaManufClusterAttributes):
    """Manufacturer specific cluster for the device."""

    """Define manufacture attributes for configuring the device."""
    attributes = {
        **TuyaMCUCluster.attributes,
        0x0274: ("voltage_coefficient", t.uint32_t, True),
        0x0275: ("current_a_coefficient", t.uint32_t, True),
        0x0276: ("power_a_coefficient", t.uint32_t, True),
        0x0277: ("energy_a_coefficient", t.uint32_t, True),
        0x027A: ("frequency_coefficient", t.uint32_t, True),
        0x027B: ("current_b_coefficient", t.uint32_t, True),
        0x027C: ("power_b_coefficient", t.uint32_t, True),
        0x027D: ("energy_b_coefficient", t.uint32_t, True),
        0x027F: ("energy_a_coefficient_reverse", t.uint32_t, True),
        0x0280: ("energy_b_coefficient_reverse", t.uint32_t, True),
        0x0281: ("report_interval", t.uint32_t, True),
    }

    dp_to_attribute = {
        101: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            ("active_power", "measurement_type"),
            converter=lambda x: (
                int(x // 10),
                0b00001111
            ),
        ),
        105: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            ("active_power", "measurement_type"),
            endpoint_id=2,
            converter=lambda x: (
                int(x // 10),
                0b00001111
            ),
        ),
        115: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            ("active_power", "measurement_type"),
            endpoint_id=3,
            converter=lambda x: (
                int(x // 10),
                0b00001001
            ),
        ),
        102: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "power_flow",
        ),
        104: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "power_flow",
            endpoint_id=2,
        ),
        106: DPToAttributeMapping(
            TuyaMetering.ep_attribute,
            "current_summ_delivered",
        ),
        107: DPToAttributeMapping(
            TuyaMetering.ep_attribute,
            "current_summ_received",
        ),
        108: DPToAttributeMapping(
            TuyaMetering.ep_attribute,
            "current_summ_delivered",
            endpoint_id=2,
        ),
        109: DPToAttributeMapping(
            TuyaMetering.ep_attribute,
            "current_summ_received",
            endpoint_id=2,
        ),
        110: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "power_factor",
        ),
        121: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "power_factor",
            endpoint_id=2,
        ),
        111: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "ac_frequency",
        ),
        112: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "rms_voltage",
        ),
        113: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "rms_current",
        ),
        114: DPToAttributeMapping(
            TuyaMatSeePlusElectricalMeasurement.ep_attribute,
            "rms_current",
            endpoint_id=2,
        ),
        116: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "voltage_coefficient",
        ),
        117: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "current_a_coefficient",
        ),
        118: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "power_a_coefficient",
        ),
        119: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "energy_a_coefficient",
        ),
        122: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "frequency_coefficient",
        ),
        123: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "current_b_coefficient",
        ),
        124: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "power_b_coefficient",
        ),
        125: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "energy_b_coefficient",
        ),
        127: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "energy_a_coefficient_reverse",
        ),
        128: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "energy_b_coefficient_reverse",
        ),
        129: DPToAttributeMapping(
            TuyaMCUCluster.ep_attribute,
            "report_interval",
        ),
    }

    data_point_handlers = {
        101: "_dp_2_attr_update",
        102: "_dp_2_attr_update",
        104: "_dp_2_attr_update",
        105: "_dp_2_attr_update",
        106: "_dp_2_attr_update",
        107: "_dp_2_attr_update",
        108: "_dp_2_attr_update",
        109: "_dp_2_attr_update",
        110: "_dp_2_attr_update",
        111: "_dp_2_attr_update",
        112: "_dp_2_attr_update",
        113: "_dp_2_attr_update",
        114: "_dp_2_attr_update",
        115: "_dp_2_attr_update",
        116: "_dp_2_attr_update",
        117: "_dp_2_attr_update",
        118: "_dp_2_attr_update",
        119: "_dp_2_attr_update",
        121: "_dp_2_attr_update",
        122: "_dp_2_attr_update",
        123: "_dp_2_attr_update",
        124: "_dp_2_attr_update",
        125: "_dp_2_attr_update",
        127: "_dp_2_attr_update",
        128: "_dp_2_attr_update",
        129: "_dp_2_attr_update",
    }

class TuyaMatSeePlusEnergyMeter2CT(EnchantedDevice):
    """Tuya MatSee Plus 2CT energy meter."""

    signature = {
        # "node_descriptor": "<NodeDescriptor byte1=1 byte2=64 mac_capability_flags=142 manufacturer_code=4417
        #                       maximum_buffer_size=66 maximum_incoming_transfer_size=66 server_mask=10752
        #                       maximum_outgoing_transfer_size=66 descriptor_capability_field=0>",
        # device_version=1
        # input_clusters=[0x0000, 0x0004, 0x0005, 0xef00]
        # output_clusters=[0x000a, 0x0019]
        MODELS_INFO: [
            ("_TZE204_81yrt3lo", "TS0601"),
            ("_TZE204_81yrt3lo", "TS0601"),
        ],
        ENDPOINTS: {
            # <SimpleDescriptor endpoint=1 profile=260 device_type=51
            # device_version=1
            # input_clusters=[0, 4, 5, 61184]
            # output_clusters=[10, 25]>
            1: {
                PROFILE_ID: zha.PROFILE_ID,
                DEVICE_TYPE: zha.DeviceType.SMART_PLUG,
                INPUT_CLUSTERS: [
                    Basic.cluster_id,
                    Groups.cluster_id,
                    Scenes.cluster_id,
                    TuyaMatSeePlusManufCluster.cluster_id,
                ],
                OUTPUT_CLUSTERS: [Time.cluster_id, Ota.cluster_id],
            },
            242: {
                # <SimpleDescriptor endpoint=242 profile=41440 device_type=97
                # input_clusters=[]
                # output_clusters=[33]
                PROFILE_ID: zgp.PROFILE_ID,
                DEVICE_TYPE: zgp.DeviceType.PROXY_BASIC,
                INPUT_CLUSTERS: [],
                OUTPUT_CLUSTERS: [GreenPowerProxy.cluster_id],
            },
        },
    }

    replacement = {
        ENDPOINTS: {
            1: {
                PROFILE_ID: zha.PROFILE_ID,
                DEVICE_TYPE: zha.DeviceType.METER_INTERFACE,
                INPUT_CLUSTERS: [
                    Basic.cluster_id,
                    Groups.cluster_id,
                    Scenes.cluster_id,
                    TuyaMatSeePlusManufCluster,
                    TuyaMatSeePlusElectricalMeasurement,
                    TuyaMetering,
                ],
                OUTPUT_CLUSTERS: [Time.cluster_id, Ota.cluster_id],
            },
            2: {
                PROFILE_ID: zha.PROFILE_ID,
                DEVICE_TYPE: zha.DeviceType.METER_INTERFACE,
                INPUT_CLUSTERS: [
                    TuyaMatSeePlusElectricalMeasurement,
                    TuyaMetering,
                ],
                OUTPUT_CLUSTERS: [],
            },
            3: {
                PROFILE_ID: zha.PROFILE_ID,
                DEVICE_TYPE: zha.DeviceType.METER_INTERFACE,
                INPUT_CLUSTERS: [
                    TuyaMatSeePlusElectricalMeasurement,
                ],
                OUTPUT_CLUSTERS: [],
            },
            242: {
                PROFILE_ID: zgp.PROFILE_ID,
                DEVICE_TYPE: zgp.DeviceType.PROXY_BASIC,
                INPUT_CLUSTERS: [],
                OUTPUT_CLUSTERS: [GreenPowerProxy.cluster_id],
            },
        }
    }
